unit batchmain;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls;

const
  numlayers = 4;

type
  TFmBatchMain = class(TForm)
    LblBatchFile: TLabel;
    EdBatchFile: TEdit;
    MmDescription: TMemo;
    BtnRunBatch: TButton;
    BtnCancelBatch: TButton;
    DlgOpenBatch: TOpenDialog;
    EdNumSpecies: TEdit;
    LblNumSpecies: TLabel;
    procedure LblBatchFileClick(Sender: TObject);
    procedure ChooseBatchFile(Sender: TObject);
    procedure BtnRunBatchClick(Sender: TObject);
    procedure BtnCancelBatchClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure EdNumSpeciesChange(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure CreateNextDriver(ofilename, dfilename: string);
    procedure CreateNextDriver4L(ofilename, dfilename: string);
  private
    { Private declarations }
  public
    { Public declarations }
  end;

var
  FmBatchMain: TFmBatchMain;
  BatchFilename:string;

implementation

{$R *.DFM}

uses stypes, frontend, fileio, Options, calculate, display;

procedure TFmBatchMain.LblBatchFileClick(Sender: TObject);
begin
 ChooseBatchFile(Sender);
 if not (batchfilename = '') then
  FmBatchMain.ActiveControl := BtnRunBatch;
end;

procedure TFmBatchMain.ChooseBatchFile(Sender: TObject);
var
 ThisEdit: TEdit;
begin
// If the user typed directly in the edit box
 if Sender is TEdit then
  begin
   ThisEdit := Sender as TEdit;
   BatchFilename:=ThisEdit.text  // Set the batch file
  end
 else  // User clicked on the label
  begin   // Show the open file dialog
   ThisEdit := ((Sender as TLabel).Focuscontrol as TEdit);
    // First set the default filename to the current file
   DlgOpenBatch.filename := BatchFilename;
    // Show the dialog and if the user clicks OK, set the new filename
   if DlgOpenBatch.execute then
    begin
      BatchFilename := DlgOpenBatch.filename;
      ThisEdit.text := Batchfilename;
    end;
  end;

 if (BatchFilename <> '') then
   if ((sender as Tcomponent).name <> 'LblBatchFilec') or
        ((sender as Tcomponent).name <> 'EdBatchFilec') then
        BtnRunBatch.Enabled := True;
end;

procedure TFmBatchMain.BtnRunBatchClick(Sender: TObject);
var
 LogFilename:string;
 temp: double;
 RunCrashed:Boolean;
begin
 MainForm.RunningInteractive := False;
 BtnRunBatch.Enabled := False;
 BtnCancelBatch.Enabled := False;
 RunCrashed := False;
 LogFilename := BatchFilename;
 ChangeExtension(LogFilename,'log');
 OpenLogFile(LogFilename, BatchFilename);
 OpenBatchFile(BatchFilename);
 try
  ReadBatchFile(paramfilename,driverfilename,outfilename,time_start,time_stop,
                   stat,FmOptions.RunOptions);
  if (Application.Title = 'sensitivity') and (paramfilename <> 'senstest.par')
    then raise Exception.Create('Invalid parameter file for sensitivity test.');
  repeat
   try
    ReadParamFile(paramfilename, ModelDef.numparam,par,
                           ModelDef.numstate,stat,temp);
    MainForm.BtnRunClick(Sender);
    WriteLogFile(LogFilename,outfilename,'Run Complete - '+ DateTimeToStr(Now));
  // single layer version
    CreateNextDriver(outfilename, driverfilename);
  // 4 layer version
//    CreateNextDriver4L(outfilename, driverfilename);
   except
    on E: Exception do
     begin
      WriteLogFile(LogFilename, outfilename, E.Message + ' - '
        + DateTimeToStr(Now));
      if not (Application.Title = 'modelbatch') then RunCrashed := True;
     end;
   end;
  until (ReadBatchFile(paramfilename,driverfilename,outfilename,
              time_start,time_stop,stat,FmOptions.RunOptions) = False) or
        (RunCrashed);
  WriteLogFile(LogFilename,batchfilename,'Batch Job Complete.');
 finally
  CloseBatchFile;
  CloseLogFile;
  BtnRunBatch.Enabled := True;
  BtnCancelBatch.Enabled := True;
  FmBatchMain.ActiveControl := BtnRunBatch;
  if RunCrashed then Abort{raise Exception.Create('Error in batch program')};
 end;
end;

procedure TFmBatchMain.BtnCancelBatchClick(Sender: TObject);
begin
 FmBatchMain.Close;
end;

procedure TFmBatchMain.FormCreate(Sender: TObject);
begin
 FmBatchMain.ActiveControl := EdBatchFile;
end;

procedure TFmBatchMain.EdNumSpeciesChange(Sender: TObject);
begin
 MainForm.SetNumSpecies(FmBatchMain.EdNumSpecies);
end;

procedure TFmBatchMain.FormShow(Sender: TObject);
begin
 if not (pos('MEL',ModelDef.modelname) > 0) then
  begin
   LblNumSpecies.Visible := False;
   LblNumSpecies.Enabled := False;
   EdNumSpecies.Visible := False;
   EdNumSpecies.Enabled := False;
  end;
end;

procedure TFmBatchMain.CreateNextDriver(ofilename, dfilename: string);
var
 catchnum, sgrownum, colLnh4d, colLno3d, colLdocd, colLdond, numRnh4u, numRno3u: integer;
 numRdonu, numRdocu, numWr, numDrd, numDrc, num, j:integer;
 tempstring, tempstring2, filename, basedrivename: string;
 Wrfile, Drdfile, Drcfile, Drlfile, oldDrive, newDrive: textfile;
 tempdrive: drivearray;
 temptime: double;

function GetColumnNumber(name:string):integer;
var
 i,colnum:integer;
 tempstring:string;
begin
 i := 1;
 colnum := 0;
 repeat
   tempstring := FmDisplayData.SgModelOutput.Cells[i,1];
   tempstring := trim(tempstring);
   if tempstring = name then colnum := i;
   i := i + 1;
 until (colnum <> 0) or (i > FmDisplayData.SgModelOutput.Colcount - 1);
 result := colnum;
end;

begin
 // Set up names and units of drive array
 tempdrive := drive;

 // Get catchment number
 tempstring := trim(ofilename);
 num := pos('cell',tempstring);
 delete(tempstring,1,num+3);
 catchnum := strtoint(tempstring[1]) + 1;    // Add 1 because calculating drivers for NEXT catchment

 // Get base driver filename
 basedrivename := dfilename;
 num := pos('cell',basedrivename);
 delete(basedrivename,num+4,length(basedrivename));

 // Read in outputfile for this catchment, get column numbers for N fluxes to downslope catchment
 // Note that I've assumed that the output file has output at the same time points as the driver!!
 // FIX
 FmDisplayData.Filename := ofilename;
 FmDisplayData.UpdateStringGrid;
 sgrownum := 3;  // 1 row for column numbers, 1 for names and 1 for units, 0 based
 colLnh4d := GetColumnNumber('N NH4 loss downslope');
 colLno3d := GetColumnNumber('N NO3 loss downslope');
 colLdocd := GetColumnNumber('C DOC loss downslope');
 colLdond := GetColumnNumber('N DON loss downslope');

 // Get array index of drivers that will be changed
 numWr := FmCalculate.GetArrayIndex(vtdriver, 'Wr');
 numDrd := FmCalculate.GetArrayIndex(vtdriver, 'Drd');
 numDrc := FmCalculate.GetArrayIndex(vtdriver, 'Drc');
 numRnh4u := FmCalculate.GetArrayIndex(vtdriver, 'Rnh4u');
 numRno3u := FmCalculate.GetArrayIndex(vtdriver, 'Rno3u');
 numRdocu := FmCalculate.GetArrayIndex(vtdriver, 'Rdocu');
 numRdonu := FmCalculate.GetArrayIndex(vtdriver, 'Rdonu');

 if (catchnum > 2) and (catchnum < 7) then
  try
 // Open Wr file
   filename := 'soil mois.xs' + inttostr(catchnum) + '1.dat';
   assignfile(Wrfile, filename);
   reset(Wrfile);

 // Open Drd file
   if catchnum < 5 then
    begin
     filename := 'lateral.xq' + inttostr(catchnum) + '1.dat';
     assignfile(Drdfile, filename);
     reset(Drdfile);
    end;

 // Open Drc file
   filename := 'lateral.xqc' + inttostr(catchnum) + '1.dat';
   assignfile(Drcfile, filename);
   reset(Drcfile);

 // Open Drl file, in this version the drainage is added to the lateral flow
   filename := 'drainage.xg' + inttostr(catchnum) + '1.dat';
   assignfile(Drlfile, filename);
   reset(Drlfile);

 // Open original driver file to get Temp, CO2, etc
   filename := dfilename;
   assignfile(OldDrive, filename);
   reset(OldDrive);

 // Create new driver file
   filename := basedrivename + inttostr(catchnum) + '.drr';
   assignfile(newDrive, filename);
   rewrite(newDrive);

 // Copy names and units from olddriver to the new driver
   readln(oldDrive, tempstring);
   writeln(NewDrive, tempstring);
   readln(oldDrive, tempstring);
   writeln(NewDrive, tempstring);

 // Advance all hydrology files to 1 Jan 1999
   for j := 1 to 1029 do
    begin
      if catchnum < 5 then readln(Drdfile, tempstring);
      readln(Drcfile, tempstring);
      readln(Drlfile, tempstring);
      readln(Wrfile, tempstring);
    end;

   while not eof(OldDrive) do
    begin
     read(oldDrive, temptime);   // Read drive time
     for j:=1 to ModelDef.numdrive do
       read(oldDrive, tempdrive[j].value);  // Read drivers
     readln(oldDrive);      // Advance to next line

    // Replace Wr, Drd and Drc with values for this catchment from hydrology model
     readln(Wrfile, tempstring);
     tempdrive[numWr].value := strtofloat(tempstring);
     if catchnum < 5 then readln(Drdfile, tempstring) else tempstring := '0';
     readln(Drlfile, tempstring2);
     tempdrive[numDrd].value := strtofloat(tempstring) + strtofloat(tempstring2);
     readln(Drcfile, tempstring);
     tempdrive[numDrc].value := strtofloat(tempstring);

     // Replace Rnh4u, Rno3u, Rdocu and Rdonu with values calculated for upslope catchment by MEL
     tempdrive[numRnh4u].value :=
         strtofloat(FmDisplayData.SgModelOutput.Cells[colLnh4d, sgrownum]);
     tempdrive[numRno3u].value :=
         strtofloat(FmDisplayData.SgModelOutput.Cells[colLno3d, sgrownum]);
     tempdrive[numRdocu].value :=
         strtofloat(FmDisplayData.SgModelOutput.Cells[colLdocd, sgrownum]);
     tempdrive[numRdonu].value :=
         strtofloat(FmDisplayData.SgModelOutput.Cells[colLdond, sgrownum]);
     sgrownum := sgrownum + 1;   // Increment row for next round

    // Write the drivers to the new driver file
     write(NewDrive,format('%-25.13e',[temptime]),' ');   // Write drive time
     for j := 1 to ModelDef.numdrive do
       write(NewDrive,format('%-25.13e',[tempdrive[j].value]),' '); // Write drivers
     writeln(NewDrive);   // Write return
    end;
  finally
   CloseFile(oldDrive);
   CloseFile(newDrive);
   if catchnum < 5 then CloseFile(DrdFile);
   CloseFile(DrcFile);
   CloseFile(DrlFile);
   CloseFile(WrFile);
  end;
end;

procedure TFmBatchMain.CreateNextDriver4L(ofilename, dfilename: string);
var
 catchnum, sgrownum, num, j, k:integer;
 colLnh4d, colLno3d, colLdocd, colLdond, numRnh4u, numRno3u, numRdonu, numRdocu,
 numWr, numDrd, numDrc, numDrl: array[1..numlayers] of integer;
 tempstring, tempstring2, filename, basedrivename: string;
 oldDrive, newDrive: textfile;
 Wrfile, Drdfile, Drcfile, Drlfile: array[1..numlayers] of textfile;
 tempdrive: drivearray;
 temptime: double;

function GetColumnNumber(name:string):integer;
var
 i,colnum:integer;
 tempstring:string;
begin
 i := 1;
 colnum := 0;
 repeat
   tempstring := FmDisplayData.SgModelOutput.Cells[i,1];
   tempstring := trim(tempstring);
   if tempstring = name then colnum := i;
   i := i + 1;
 until (colnum <> 0) or (i > FmDisplayData.SgModelOutput.Colcount - 1);
 result := colnum;
end;

begin
 // Set up names and units of drive array
 tempdrive := drive;

 // Get catchment number
 tempstring := trim(ofilename);
 num := pos('cell', tempstring);
 delete(tempstring, 1, num+3);
 catchnum := strtoint(tempstring[1]) + 2;    // Add 2 because calculating drivers for every other catchment
                                             // i.e. 1 flows to 3, 2 to 4, etc.
 // Get base driver filename
 basedrivename := dfilename;
 num := pos('cell', basedrivename);
 delete(basedrivename, num+4, length(basedrivename));

 // Read in outputfile for this catchment, get column numbers for N fluxes to downslope catchment
 // Note that I've assumed that the output file has output at the same time points as the driver!!
 // FIX
 FmDisplayData.Filename := ofilename;
 FmDisplayData.UpdateStringGrid;
 sgrownum := 3;  // 1 row for column numbers, 1 for names and 1 for units, 0 based
 for j := 1 to numlayers do
  begin
   colLnh4d[j] := GetColumnNumber('*N NH4 loss downslope' + inttostr(j));
   colLno3d[j] := GetColumnNumber('*N NO3 loss downslope' + inttostr(j));
   colLdocd[j] := GetColumnNumber('*C DOC loss downslope' + inttostr(j));
   colLdond[j] := GetColumnNumber('*N DON loss downslope' + inttostr(j));

 // Get array index of drivers that will be changed
   numWr[j] := FmCalculate.GetArrayIndex(vtdriver, 'Wr' + '[' + inttostr(j) + ']');
   numDrd[j] := FmCalculate.GetArrayIndex(vtdriver, 'Drd' + '[' + inttostr(j) + ']');
   numDrc[j] := FmCalculate.GetArrayIndex(vtdriver, 'Drc' + '[' + inttostr(j) + ']');
   numDrl[j] := FmCalculate.GetArrayIndex(vtdriver, 'Drl' + '[' + inttostr(j) + ']');
   numRnh4u[j] := FmCalculate.GetArrayIndex(vtdriver, 'Rnh4u' + '[' + inttostr(j) + ']');
   numRno3u[j] := FmCalculate.GetArrayIndex(vtdriver, 'Rno3u' + '[' + inttostr(j) + ']');
   numRdocu[j] := FmCalculate.GetArrayIndex(vtdriver, 'Rdocu' + '[' + inttostr(j) + ']');
   numRdonu[j] := FmCalculate.GetArrayIndex(vtdriver, 'Rdonu' + '[' + inttostr(j) + ']');
  end;

 if (catchnum > 2) and (catchnum < 7) then
  try
   for j := 1 to numlayers do
    begin
 // Open Wr file
     filename := 'soil mois.xs' + inttostr(catchnum) + inttostr(j) + '.dat';
     assignfile(Wrfile[j], filename);
     reset(Wrfile[j]);

 // Open Drd file
     if catchnum < 5 then
      begin
       filename := 'lateral.xq' + inttostr(catchnum) + inttostr(j) + '.dat';
       assignfile(Drdfile[j], filename);
       reset(Drdfile[j]);
      end;

 // Open Drc file
     filename := 'lateral.xqc' + inttostr(catchnum) + inttostr(j) + '.dat';
     assignfile(Drcfile[j], filename);
     reset(Drcfile[j]);

 // Open Drl file
     if j <> numlayers then
      begin
       filename := 'drainage.xg' + inttostr(catchnum) + inttostr(j) + '.dat';
       assignfile(Drlfile[j], filename);
       reset(Drlfile[j]);
      end;
   end;

 // Open original driver file to get Temp, CO2, etc
   filename := dfilename;
   assignfile(OldDrive, filename);
   reset(OldDrive);

 // Create new driver file
   filename := basedrivename + inttostr(catchnum) + '.drr';
   assignfile(newDrive, filename);
   rewrite(newDrive);

 // Copy names and units from olddriver to the new driver
   readln(oldDrive, tempstring);
   writeln(NewDrive, tempstring);
   readln(oldDrive, tempstring);
   writeln(NewDrive, tempstring);

 // Advance all hydrology files to 1 Jan 1999
   for j := 1 to 1029 do
    for  k := 1 to numlayers do
     begin
      readln(Wrfile[k], tempstring);
      if catchnum < 5 then readln(Drdfile[k], tempstring);
      readln(Drcfile[k], tempstring);
      if k <> numlayers then readln(Drlfile[k], tempstring);
     end;

   while not eof(OldDrive) do
    begin
     read(oldDrive, temptime);   // Read drive time
     for j:=1 to ModelDef.numdrive do
       read(oldDrive, tempdrive[j].value);  // Read drivers
     readln(oldDrive);      // Advance to next line

    // Replace Wr, Drd, Drc and Drl with values for this catchment from hydrology model
     for j := 1 to numlayers do
      begin
       readln(Wrfile[j], tempstring);
       tempdrive[numWr[j]].value := strtofloat(tempstring);
       if catchnum < 5 then readln(Drdfile[j], tempstring) else tempstring := '0';
       tempdrive[numDrd[j]].value := strtofloat(tempstring);
       if j <> numlayers then readln(Drlfile[j], tempstring) else tempstring := '0';
       tempdrive[numDrl[j]].value := strtofloat(tempstring);
       readln(Drcfile[j], tempstring);
       tempdrive[numDrc[j]].value := strtofloat(tempstring);

     // Replace Rnh4u, Rno3u, Rdocu and Rdonu with values calculated for upslope catchment by MEL
       tempdrive[numRnh4u[j]].value :=
         strtofloat(FmDisplayData.SgModelOutput.Cells[colLnh4d[j], sgrownum]);
       tempdrive[numRno3u[j]].value :=
         strtofloat(FmDisplayData.SgModelOutput.Cells[colLno3d[j], sgrownum]);
       tempdrive[numRdocu[j]].value :=
         strtofloat(FmDisplayData.SgModelOutput.Cells[colLdocd[j], sgrownum]);
       tempdrive[numRdonu[j]].value :=
         strtofloat(FmDisplayData.SgModelOutput.Cells[colLdond[j], sgrownum]);
      end;
     sgrownum := sgrownum + 1;   // Increment row for next round

    // Write the drivers to the new driver file
     write(NewDrive,format('%-25.13e',[temptime]),' ');   // Write drive time
     for j := 1 to ModelDef.numdrive do
       write(NewDrive,format('%-25.13e',[tempdrive[j].value]),' '); // Write drivers
     writeln(NewDrive);   // Write return
    end;
  finally
   CloseFile(oldDrive);
   CloseFile(newDrive);
   for j := 1 to numlayers do
    begin
     CloseFile(WrFile[j]);
     if catchnum < 5 then CloseFile(DrdFile[j]);
     CloseFile(DrcFile[j]);
     if j <> numlayers then CloseFile(DrlFile[j]);
    end;
  end;
end;

end.
